/************************************************************************************************************\

Module Name:    AinModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "AinModule.h"
#include "LBoard.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

static const MCHAR8     g_szModuleNameBase[] = "Ain";

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

void AinMod_DestroyBuffers(AinModule* poAinMod);
MBOOL AinMod_IsSourceModeValid(AinModule *poAinMod);

/************************************************************************************************************\

Function:       AinMod_Init

Description:

\************************************************************************************************************/
LStatus AinMod_Init(
    AinModule*          poAinMod,
    LDevice_Handle      hDevice,
    MUINT32             uiAinIndex,
    MUINT32             uiBufferCount,
    MUINT32             uiFramesPerBuffer)
{
    MsgLog(2, "{...");

    AinMod_Cleanup(poAinMod);

    LStatus eStatus = ((poAinMod != MNULL) && (hDevice != MNULL))
                      ? LStatus_OK
                      : LStatus_INVALID_PARAM;

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        snprintf(
                    poAinMod->szModuleName,
                    sizeof(poAinMod->szModuleName),
                    "%s%d",
                    g_szModuleNameBase,
                    uiAinIndex);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LBoard_GetTickReferenceFrequency(hDevice, &(poAinMod->uiTickFrequency));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LAudioIn_GetHandle(
                      hDevice,
                      uiAinIndex,
                      LAccessMode_READWRITE_EXCLUSIVE,
                      &(poAinMod->hAin));
    }

    eStatus = AinMod_GetSourceParams(poAinMod);

    MUINT32 uiBufferSize = 0;

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LAudioFormat_ComputeBufferSizeBasedOnFrame(
                      poAinMod->eAudioFormat,
                      uiFramesPerBuffer,
                      &uiBufferSize);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LBuffer_Attributes oBufAttrib;

        if (poAinMod->bFailSafe)
        {
            oBufAttrib.eAttributeType                           = LBuffer_Type_SYSTEM_LINEAR;
            oBufAttrib.oSystemLinearAttributes.pvSystemMemory   = MNULL;
            oBufAttrib.oSystemLinearAttributes.uiSize           = uiBufferSize;
        }

        eStatus = ModLnk_Init(
                    &(poAinMod->oOutLink),
                    hDevice,
                    uiBufferCount,
                    poAinMod->bFailSafe ? &oBufAttrib : MNULL,
                    !poAinMod->bFailSafe,
                    0,
                    poAinMod->szModuleName);

        if (poAinMod->bFailSafe && LSTATUS_IS_SUCCESS(eStatus))
        {
            MINT i;
            for (i = 0; (i < poAinMod->oOutLink.uiBufferCount) && LSTATUS_IS_SUCCESS(eStatus); i++)
            {
                LBuffer_Handle hBuffer = poAinMod->oOutLink.aoBufferInfo[i].hBuffer;
                MUINT8* puiBuffer = MNULL;

                eStatus = LBuffer_BeginAccess(hBuffer, 0, 1, &puiBuffer);

                if(LSTATUS_IS_SUCCESS(eStatus))
                {
                    memset(puiBuffer, 0, uiBufferSize);
                    LBuffer_EndAccess(hBuffer);
                }
            }
        }
    }
    else
    {
        MsgLog(0, "Failed to detect audio source.");
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        poAinMod->bSourceParamsValid    = MTRUE;
        poAinMod->uiFramesPerBuffer     = uiFramesPerBuffer;
        poAinMod->uiBufferSize          = uiBufferSize;
    }
    else
    {
        AinMod_Cleanup(poAinMod);
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       AinMod_Cleanup

Description:    .

\************************************************************************************************************/
void AinMod_Cleanup(AinModule* poAinMod)
{
    if (poAinMod != MNULL)
    {
        AinMod_DestroyBuffers(poAinMod);
        ModLnk_Cleanup(&poAinMod->oOutLink);

        if(poAinMod->hAin != MNULL)
        {
            LAudioIn_ReleaseHandle(poAinMod->hAin);
            poAinMod->hAin = MNULL;
        }

        poAinMod->bSourceParamsValid = MFALSE;
        poAinMod->bFailSafe = MFALSE;
    }
}

/************************************************************************************************************\

Function:       AinMod_CpuThread

Description:    .

\************************************************************************************************************/
LStatus AinMod_CpuThread(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    AinModule* poAinMod      = (AinModule*)pvData;
    MUINT64    uiPrevTickRef = ~0;

    ModThread_SetName(poAinMod->szModuleName);
    MsgLog(2, "Start thread %p.", pthread_self());

    LStatus eStatus = poAinMod->bFailSafe ? LStatus_OK : LAudioIn_StartCapture(poAinMod->hAin);

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        const MUINT64 uiStartTimeUsec = GetMonoTimeUsec();

        poAinMod->uiElapsedTimeUsec     = 0;
        poAinMod->uiTotalBufferCount    = 0;
        poAinMod->uiCapturedBufferCount = 0;

        while (!poAinMod->oCpuThread.bKillThread)
        {
            if (!AinMod_IsSourceModeValid(poAinMod))
            {
                MsgLog(2, "Mode changed. Stop capturing.");
                poAinMod->oCpuThread.bKillThread = MTRUE;
                break;
            }

            BufferInfo* poBuffer = MNULL;

            eStatus = ModLnk_GetReturnedBuffer(&(poAinMod->oOutLink), 100, MNULL, &poBuffer);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                MUINT64 uiBufferCount   = poAinMod->uiTotalBufferCount + 1;
                MUINT64 uiTickRef       = 0;
                MUINT32 uiPayloadSize   = 0;

                if (!poAinMod->bFailSafe)
                {
                    if (poBuffer->hBuffer != MNULL)
                    {
                        MsgLog(4, "LAudioIn_ReleaseBuffer(Buffer[%u])...", poBuffer->uiId);
                        LAudioIn_ReleaseBuffer(poAinMod->hAin, poBuffer->hBuffer);
                        poBuffer->hBuffer = MNULL;
                    }

                    MsgLog(4, "LAudioIn_GetNextBuffer(Buffer[%u])...", poBuffer->uiId);

                    eStatus = LAudioIn_GetNextBuffer(
                                    poAinMod->hAin,
                                    MTRUE,
                                    100,
                                    &(poBuffer->hBuffer),
                                    &uiPayloadSize,
                                    &uiBufferCount,
                                    &uiTickRef);

                    MsgLog(
                        4,
                        "LAudioIn_GetNextBuffer done. (Size=%u, BufferCount= %lu, TickRef= %lu, status= %d - %s) "
                        "Got count = %u",
                        uiPayloadSize,
                        uiBufferCount,
                        uiTickRef,
                        eStatus,
                        GetStatusStr(eStatus),
                        poAinMod->uiCapturedBufferCount);
                }

                poAinMod->uiElapsedTimeUsec = GetMonoTimeUsec() - uiStartTimeUsec;

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    if (poAinMod->bFailSafe)
                    {
                        MUINT64 uiTimestampUsec = (uiBufferCount * 1000 * 1000)
                                                  / LAudioFormat_GetSampleRate(poAinMod->eAudioFormat);

                        if (uiTimestampUsec > poAinMod->uiElapsedTimeUsec)
                        {
                            usleep(uiTimestampUsec - poAinMod->uiElapsedTimeUsec);
                        }

                        poBuffer->uiTimestampUsec = GetMonoTimeUsec();
                    }
                    else
                    {
                        poBuffer->uiTimestampUsec = (uiTickRef * 1000 * 1000) / poAinMod->uiTickFrequency;
                    }

                    poBuffer->uiSyncPtsUsec   = poBuffer->uiTimestampUsec;
                    poBuffer->uiStartOffset   = 0;
                    poBuffer->uiSizeBytes     = poAinMod->uiBufferSize;

                    poAinMod->uiTotalBufferCount = uiBufferCount;
                    poAinMod->uiCapturedBufferCount++;

                    if (uiPrevTickRef != ~0)
                    {
                        // Check tick reference
                        if (uiTickRef < uiPrevTickRef)
                        {
                            MsgLog(0, "WARNING! New tick reference (%lu) smaller than the new one (%lu).",
                                      uiTickRef, uiPrevTickRef);
                        }
                        else if ((uiTickRef - uiPrevTickRef) > 2*1000*1000)
                        {
                            MsgLog(0, "WARNING! Big difference between two tick refs: (%lu - %lu)/%u sec.",
                                      uiTickRef, uiPrevTickRef, poAinMod->uiTickFrequency);
                        }
                    }

                    uiPrevTickRef = uiTickRef;

                    ModLnk_SubmitBuffer(&(poAinMod->oOutLink), poBuffer, MNULL, NO_TAG);
                }
                else
                {
                    ModLnk_ReleaseBuffer(&(poAinMod->oOutLink), poBuffer);
                }
            }

            if (LSTATUS_IS_FAIL(eStatus)
                && (eStatus != LStatus_TIMEOUT))
            {
                usleep(1000);
            }
        }

        if (!poAinMod->bFailSafe)
        {
            MsgLog(2, "Stopping capture.");
            LAudioIn_StopCapture(poAinMod->hAin);
        }
    }
    else
    {
        MsgLogErr("ERROR! LAudioIn_StartCapture returned %d", eStatus);
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       AinMod_IsSourceModeValid

Description:    .

\************************************************************************************************************/
MBOOL AinMod_IsSourceModeValid(AinModule* poAinMod)
{
    if (poAinMod->hAin == MNULL)
    {
        return MFALSE;
    }

    LAudioIn_Event eEvent;

    while (LSTATUS_IS_SUCCESS(LAudioIn_WaitForEvent(poAinMod->hAin, 0, &eEvent)))
    {
        if (eEvent == LAudioIn_Event_SOURCE_CHANGED)
        {
            poAinMod->bSourceParamsValid = MFALSE;
        }
    }

    return poAinMod->bSourceParamsValid;
}

/************************************************************************************************************\

Function:       AinMod_GetSourceParams

Description:    .

\************************************************************************************************************/
LStatus AinMod_GetSourceParams(AinModule* poAinMod)
{
    LStatus eStatus = ((poAinMod != MNULL) && (poAinMod->hAin != MNULL))
                      ? LStatus_OK
                      : LStatus_FAIL;

    MsgLog(2, "AinMod_GetSourceParams...");

    while (LSTATUS_IS_SUCCESS(eStatus)
           && !AinMod_IsSourceModeValid(poAinMod))
    {
        MBOOL   bIsCapturable = MFALSE;
        MBOOL   bIsDetected   = MFALSE;

        eStatus = LAudioIn_DetectSource(poAinMod->hAin, &bIsCapturable, &bIsDetected);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (bIsCapturable && bIsDetected)
            {
                LAudioFormat    eAudioFmt;
                MINT            i = 0;

                LAudioFormat_Descriptor oDesc;
                memset(&oDesc, 0, sizeof(oDesc));

                while (LSTATUS_IS_SUCCESS(LAudioIn_EnumSupportedAudioFormat(poAinMod->hAin, i, &eAudioFmt)))
                {
                    // Keep the first supported format.
                    if (i == 0)
                    {
                        poAinMod->eAudioFormat = eAudioFmt;
                        poAinMod->bSourceParamsValid = MTRUE;
                    }

                    MsgLog(2, "Suported audio format [%u] = 0x%08X", i, eAudioFmt);
                    i++;
                }
            }
            else
            {
                MsgLog(2, "WARNING! Audio signal not detected(%d) nor capturable(%d). Go in FailSafe mode.",
                          bIsDetected, bIsCapturable);

                poAinMod->bFailSafe             = MTRUE;
                poAinMod->eAudioFormat          = LAudioFormat_STEREO_16_48000HZ;
                poAinMod->bSourceParamsValid    = MTRUE;
            }
        }
        else
        {
            MsgLogErr(
                "ERROR! LAudioIn_DetectSource returned %d (%s).",
                eStatus,
                GetStatusStr(eStatus));
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       AinMod_Start

Description:    .

\************************************************************************************************************/
LStatus AinMod_Start(AinModule* poAinMod)
{
    MsgLog(2, "{...");

    LStatus eStatus = ((poAinMod != MNULL) && (poAinMod->hAin != MNULL))
                      ? LStatus_OK
                      : LStatus_INVALID_PARAM;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        if (poAinMod->oOutLink.uiSubmitCount == 0)
        {
            MsgLogErr("ERROR! Bad connection.");
            eStatus = LStatus_FAIL;
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        AinMod_DestroyBuffers(poAinMod);

        if (!poAinMod->bFailSafe)
        {
            MUINT32 uiNbBuffers = poAinMod->oOutLink.uiBufferCount + 4;
            MsgLog(2, "LAudioIn_CreateBuffers(AudFmt= 0x%08X, Count= %u)...",
                        poAinMod->eAudioFormat, uiNbBuffers);

            MUINT32 uiBufferSize = 0;
            eStatus = LAudioFormat_ComputeBufferSizeBasedOnFrame(
                          poAinMod->eAudioFormat,
                          poAinMod->uiFramesPerBuffer,
                          &uiBufferSize);

            if(LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = LAudioIn_CreateBuffers(
                            poAinMod->hAin,
                            poAinMod->eAudioFormat,
                            uiBufferSize,
                            uiNbBuffers);

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    MsgLog(2, "LAudioIn_CreateBuffers done.");
                    poAinMod->bInternalBufferCreated = MTRUE;
                }
                else
                {
                    MsgLogErr("ERROR! LAudioIn_CreateBuffers returned %d.", eStatus);
                }
            }
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = ModThread_Start(&(poAinMod->oCpuThread), poAinMod, AinMod_CpuThread);
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       AinMod_Stop

Description:    .

\************************************************************************************************************/
void AinMod_Stop(AinModule* poAinMod)
{
    MsgLog(2, "{...");

    if (poAinMod != MNULL)
    {
        ModThread_Stop(&(poAinMod->oCpuThread));
    }

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       AinMod_DestroyBuffers

Description:    .

\************************************************************************************************************/
void AinMod_DestroyBuffers(AinModule* poAinMod)
{
    if (poAinMod != MNULL)
    {
        if (poAinMod->bInternalBufferCreated)
        {
            MUINT i;

            for (i = 0; i < poAinMod->oOutLink.uiBufferCount; i++)
            {
                BufferInfo* poBufferInfo = &(poAinMod->oOutLink.aoBufferInfo[i]);

                if (poBufferInfo->hBuffer != MNULL)
                {
                    LAudioIn_ReleaseBuffer(poAinMod->hAin,  poBufferInfo->hBuffer);
                    poBufferInfo->hBuffer = MNULL;
                }
            }

            LAudioIn_DestroyBuffers(poAinMod->hAin);
            poAinMod->bInternalBufferCreated = MFALSE;
        }
    }
}
